## Authentication Credentials

```json
{
  "email": "mtn_payment@tcs.com",
  "password": "tcs@mtn123"
}
```

After successful authentication, an `access_token` is returned and used as follows:

```http
Authorization: Bearer {access_token}
```

---

## Important MTN Environment Variables

```env
MTN_TERMINAL_ID=9001000000062075
MTN_BASE_URL=https://cashmobile.mtnsyr.com:9000
MTN_TTL=30
```
----------------------------------------------X-SIGNATURE----------------------------------------------
# X-SIGNATURE Mechanism in MTN Payment API

## What is X-SIGNATURE?

`X-SIGNATURE` is a **digital security signature** sent in the HTTP request header to ensure:

* The request is coming from a **trusted source**
* The request payload **has not been tampered with**
* Protection of the API against **forged or replayed requests**

The signature is generated using **HMAC SHA256** with a **shared secret key** between the client and the server.

---

## General Workflow

1. Prepare the request **Payload**
2. Convert the payload to **JSON**
3. Generate a signature using:

   * `sha256` hashing algorithm
   * The secret key `API_SIGNATURE_SECRET`
4. Send the generated signature in the request header as `X-SIGNATURE`
5. The server regenerates the signature using the same method
6. The server compares both signatures:

   * If they match ➜ request is accepted ✅
   * If they do not match ➜ request is rejected ❌

---

## Secret Key

The secret key must be stored securely in the `.env` file and **must never be shared**.

```env
API_SIGNATURE_SECRET=super-secret-key-123456
```

⚠️ **Important:**
The secret key itself must never be sent in the request.
Only the generated signature is sent.

---

## Practical Example (Laravel)

### Code Example

```php
Route::get('/test', function () {
    // Request payload
    $payload = [
        'invoice' => 101,
    ];
    
    // Generate signature
    $signature = hash_hmac(
        'sha256',
        json_encode($payload),
        env('API_SIGNATURE_SECRET')
    );
    
    // Send request with signature
    $response = Http::withHeaders([
            'X-SIGNATURE' => $signature,
            'Authorization' => 'Bearer 35|GwP76Ec0OupkIUqNnBgJJJnRk6BzAqJcXzhYdoQ645415e39',
            'Accept' => 'application/json',
        ])
        ->post('http://127.0.0.1:8001/api/mtn/v1/invoice/get', $payload);

    return $response->json();
        
});
```

---

## Code Explanation (Step by Step)

### 1. Preparing the Payload

```php
$payload = [
    'invoice' => 101,
];
```

This is the data that will be sent to the API.

---

### 2. Converting the Payload to JSON

```php
json_encode($payload)
```

⚠️ The **exact same structure and order** must be used when generating and validating the signature on the server.

---

### 3. Generating the Signature

```php
$signature = hash_hmac(
    'sha256',
    json_encode($payload),
    env('API_SIGNATURE_SECRET')
);
```

* `sha256` → hashing algorithm
* `json_encode($payload)` → request data
* `API_SIGNATURE_SECRET` → shared secret key

The result is an encrypted string representing the digital signature.

---

### 4. Sending the Signature in the Header

```http
X-SIGNATURE: generated_signature_here
```

---

## Server-Side Validation Process

When the server receives the request:

1. It reads the request payload
2. Regenerates the signature using the same method
3. Compares the generated signature with the received one
4. If the signatures do not match:

   * The server returns an error (401 or 403)
   * The request is rejected

---

## Relationship Between X-SIGNATURE and Authorization

| Component     | Purpose                                     |
| ------------- | ------------------------------------------- |
| Authorization | Verifies the user identity (Bearer Token)   |
| X-SIGNATURE   | Verifies request integrity and authenticity |

➡️ **Both headers are required** for a successful request.

---

## Authentication Credentials

```json
{
  "email": "mtn_payment@tcs.com",
  "password": "tcs@mtn123"
}
```

After successful authentication, an `access_token` is returned and used as follows:

```http
Authorization: Bearer {access_token}
```

---

## Important MTN Environment Variables

```env
MTN_TERMINAL_ID=9001000000062075
MTN_BASE_URL=https://cashmobile.mtnsyr.com:9000
MTN_TTL=30
```

---

## Summary

* `X-SIGNATURE` provides an additional security layer
* It relies on a shared **secret key**
* Any modification to the request data causes signature validation to fail
* It is essential for protecting payment and invoicing operations

---

📌 **Note:**
The same signature mechanism must be applied to all API endpoints, including:

* `invoice/create`
* `invoice/get`
* `payment/initiate`
* `payment/confirm`
