<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use App\Models\User;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\RateLimiter;
use Laravel\Sanctum\PersonalAccessToken;
use Exception;

class AuthController extends ApiController
{

    /**
     * Maximum login attempts
     */
    private const MAX_LOGIN_ATTEMPTS = 5;

    /**
     * Login rate limit duration (minutes)
     */
    private const LOGIN_RATE_LIMIT = 15;

    /**
     * Maximum refresh attempts
     */
    private const MAX_REFRESH_ATTEMPTS = 10;

    /**
     * Refresh rate limit duration (minutes)
     */
    private const REFRESH_RATE_LIMIT = 5;

    /**
     * Handle user login
     */
    public function login(Request $request)
    {
        try {
            // Check rate limiting for login
            $this->checkLoginRateLimit($request);

            // Validate request
            $validated = $request->validate([
                'email' => 'required|email|max:255',
                'password' => 'required|string|min:6',
                'device_name' => 'nullable|string|max:255',
            ]);

            // Attempt authentication
            if (!Auth::attempt([
                'email' => $validated['email'],
                'password' => $validated['password']
            ])) {
                // Increment failed attempts
                $this->incrementFailedAttempts($request);

                return $this->unauthorized('Invalid credentials. Please check your email and password.');
            }

            // Clear failed attempts on successful login
            $this->clearFailedAttempts($request);

            $user = Auth::user();

            // Check if user is active
            if (!$this->isUserActive($user)) {
                Auth::logout();
                return $this->forbidden('Your account is not active. Please contact support.');
            }

            // Create tokens
            $deviceName = $validated['device_name'] ?? $request->header('User-Agent', 'Unknown Device');
            // $accessToken = $user->createToken('access_token', ['*'], now()->addMinutes(config('sanctum.expiration', 60)))->plainTextToken;
            // $refreshToken = $user->createToken('refresh_token', ['refresh'], now()->addDays(30))->plainTextToken;

            $accessToken = $user->createToken('access_token', ['access'], now()->addMinutes(60))->plainTextToken;
            $refreshToken = $user->createToken('refresh_token', ['refresh'], now()->addDays(30))->plainTextToken;

            // Log successful login
            Log::info('User logged in', [
                'user_id' => $user->id,
                'email' => $user->email,
                'device' => $deviceName,
                'ip' => $request->ip()
            ]);

            return $this->success([
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'email_verified_at' => $user->email_verified_at,
                ],
                'tokens' => [
                    'access_token' => $accessToken,
                    'refresh_token' => $refreshToken,
                    'token_type' => 'Bearer',
                    'expires_in' => config('sanctum.expiration', 60) * 60,
                    'refresh_token_expires_in' => 30 * 24 * 60 * 60 // 30 days in seconds
                ],
                'device' => $deviceName
            ], 'Login successful');
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'email' => $request->email,
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }

    /**
     * Handle token refresh
     */
    public function refresh(Request $request)
    {
        try {
            // Check rate limiting for refresh
            $this->checkRefreshRateLimit($request);

            // Validate request
            $validated = $request->validate([
                'refresh_token' => 'required|string',
                'device_name' => 'nullable|string|max:255',
            ]);

            $refreshToken = $validated['refresh_token'];

            // Extract token ID safely
            $tokenParts = explode('|', $refreshToken);
            if (count($tokenParts) < 2) {
                $this->incrementRefreshAttempts($request);
                return $this->unauthorized('Invalid refresh token format');
            }

            // Find the token in database
            $tokenModel = PersonalAccessToken::findToken($refreshToken);

            if (!$tokenModel) {
                $this->incrementRefreshAttempts($request);
                return $this->unauthorized('Invalid or expired refresh token');
            }

            // Check if token has refresh ability
            if (!$tokenModel->can('refresh')) {
                $this->incrementRefreshAttempts($request);
                return $this->unauthorized('Token cannot be used for refresh');
            }

            // Check if refresh token is expired
            if ($tokenModel->expires_at && $tokenModel->expires_at->isPast()) {
                $tokenModel->delete();
                $this->incrementRefreshAttempts($request);
                return $this->unauthorized('Refresh token has expired');
            }

            $user = $tokenModel->tokenable;

            // Check if user is active
            if (!$this->isUserActive($user)) {
                $tokenModel->delete();
                return $this->forbidden('Your account is not active');
            }

            // Delete the old refresh token (token rotation)
            $tokenModel->delete();

            // Create new tokens
            $deviceName = $validated['device_name'] ?? $request->header('User-Agent', 'Unknown Device');
            // $newAccessToken = $user->createToken('access_token', ['*'], now()->addMinutes(config('sanctum.expiration', 60)))->plainTextToken;
            // $newRefreshToken = $user->createToken('refresh_token', ['refresh'], now()->addDays(30))->plainTextToken;

            $newAccessToken = $user->createToken('access_token', ['access'], now()->addMinutes(60))->plainTextToken;
            $newRefreshToken = $user->createToken('refresh_token', ['refresh'], now()->addDays(30))->plainTextToken;


            // Clear refresh attempts on success
            $this->clearRefreshAttempts($request);

            // Log token refresh
            Log::info('Token refreshed', [
                'user_id' => $user->id,
                'email' => $user->email,
                'device' => $deviceName,
                'ip' => $request->ip()
            ]);

            return $this->success([
                'tokens' => [
                    'access_token' => $newAccessToken,
                    'refresh_token' => $newRefreshToken,
                    'token_type' => 'Bearer',
                    'expires_in' => config('sanctum.expiration', 60) * 60,
                    'refresh_token_expires_in' => 30 * 24 * 60 * 60
                ],
                'device' => $deviceName
            ], 'Token refreshed successfully');
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Token refresh error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during token refresh');
        }
    }

    /**
     * Handle user logout
     */
    public function logout(Request $request)
    {
        // dd();
        try {
            $user = $request->user();
            $currentToken = $user->currentAccessToken();


            // Log logout
            Log::info('User logged out', [
                'user_id' => $user->id,
                'email' => $user->email,
                'token_id' => $currentToken->id,
                'ip' => $request->ip()
            ]);

            // Delete current token
            $currentToken->delete();

            return $this->success(null, 'Successfully logged out');
        } catch (Exception $e) {
            Log::error('Logout error', [
                'error' => $e->getMessage(),
                'user_id' => $request->user()->id ?? 'unknown',
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during logout');
        }
    }

    /**
     * Handle logout from all devices
     */
    public function logoutAll(Request $request)
    {
        try {
            $user = $request->user();
            $tokenCount = $user->tokens()->count();

            // Log logout all
            Log::info('User logged out from all devices', [
                'user_id' => $user->id,
                'email' => $user->email,
                'tokens_deleted' => $tokenCount,
                'ip' => $request->ip()
            ]);

            // Delete all tokens
            $user->tokens()->delete();

            return $this->success([
                'tokens_deleted' => $tokenCount
            ], 'Successfully logged out from all devices');
        } catch (Exception $e) {
            Log::error('Logout all error', [
                'error' => $e->getMessage(),
                'user_id' => $request->user()->id ?? 'unknown',
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during logout from all devices');
        }
    }

    /**
     * Get current authenticated user
     */
    public function me(Request $request)
    {
        try {
            $user = $request->user();

            return $this->success([
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'email_verified_at' => $user->email_verified_at,
                    'created_at' => $user->created_at,
                    'updated_at' => $user->updated_at,
                ]
            ], 'User retrieved successfully');
        } catch (Exception $e) {
            Log::error('Get user error', [
                'error' => $e->getMessage(),
                'user_id' => $request->user()->id ?? 'unknown'
            ]);

            return $this->serverError('An error occurred while retrieving user information');
        }
    }

    /**
     * Rate limiting methods
     */
    private function checkLoginRateLimit(Request $request): void
    {
        $key = 'login_attempts:' . $request->ip();

        if (RateLimiter::tooManyAttempts($key, self::MAX_LOGIN_ATTEMPTS)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'email' => "Too many login attempts. Please try again in {$seconds} seconds."
            ])->status(429);
        }
    }

    private function incrementFailedAttempts(Request $request): void
    {
        $key = 'login_attempts:' . $request->ip();
        RateLimiter::hit($key, self::LOGIN_RATE_LIMIT * 60);
    }

    private function clearFailedAttempts(Request $request): void
    {
        $key = 'login_attempts:' . $request->ip();
        RateLimiter::clear($key);
    }

    private function checkRefreshRateLimit(Request $request): void
    {
        $key = 'refresh_attempts:' . $request->ip();

        if (RateLimiter::tooManyAttempts($key, self::MAX_REFRESH_ATTEMPTS)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'refresh_token' => "Too many refresh attempts. Please try again in {$seconds} seconds."
            ])->status(429);
        }
    }

    private function incrementRefreshAttempts(Request $request): void
    {
        $key = 'refresh_attempts:' . $request->ip();
        RateLimiter::hit($key, self::REFRESH_RATE_LIMIT * 60);
    }

    private function clearRefreshAttempts(Request $request): void
    {
        $key = 'refresh_attempts:' . $request->ip();
        RateLimiter::clear($key);
    }

    /**
     * Check if user is active
     */
    private function isUserActive(User $user): bool
    {
        // Add your custom user status checks here
        // Example: check if user is banned, suspended, etc.

        // Check if email is verified (optional)
        if (config('auth.must_verify_email') && !$user->hasVerifiedEmail()) {
            return false;
        }

        // Check if user is active (you might have an 'is_active' column)
        if (property_exists($user, 'is_active') && !$user->is_active) {
            return false;
        }

        return true;
    }
}
