<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\MtnApiService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class MtnController extends ApiController
{
    public function activateTerminal(Request $request)
    {
        try {
            $request->validate([
                'terminal_id' => 'required|string',
                'activation_code' => 'required|string|size:8',
                'serial_number' => 'sometimes|string',
                'key_size' => 'sometimes|integer|in:1024,2048,4096'
            ]);
            $service = new MtnApiService(
                $request->terminal_id,
                $request->serial_number
            );

            $service->generateKeys($request->key_size ?? 1024);
            $result = $service->activateTerminal($request->activation_code);


            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function createInvoice(Request $request)
    {
        try {
            $request->validate([
                'amount' => 'required|numeric|min:0.1',
                'invoice' => 'required|integer',
                'ttl' => 'sometimes|integer|min:1',
            ]);

            $amountMinorUnits = $request->amount * 100;
            $data = [
                'Amount' => (float)$amountMinorUnits,
                'Invoice' => (int)$request->invoice,
                'TTL' => (int)$request->ttl  ?? config('app.ttl'),
            ];

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();
            $result = $service->hashDataAndSendMtn($data, 'create_invoice', 'pos_web/invoice/create');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function getInvoice(Request $request)
    {
        try {

            $request->validate([
                'invoice' => 'required|integer',
            ]);

            $data = ["invoice" => (int)$request->invoice];

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();
            $result = $service->hashDataAndSendMtn($data, 'get_invoice', 'pos_web/invoice/get');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function initiatePayment(Request $request)
    {
        try {
            $request->validate([
                'invoice' => 'required|integer',
                'phone_clinte' => 'required|string',
                'guid' => 'required|string',
            ]);

            $data = ["Invoice" => (int)$request->invoice, "Phone" => $request->phone_clinte, "Guid" => $request->guid];

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();
            $result = $service->hashDataAndSendMtn($data, 'inti_invoice', 'pos_web/payment_phone/initiate');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function confirmPayment(Request $request)
    {
        try {
            $request->validate([
                'phone' => 'required|string',
                'guid' => 'required|string',
                'operation_number' => 'required|integer',
                'invoice' => 'required|integer',
                'code' => 'required|string',
            ]);

            $code = base64_encode(hash('sha256', $request->code, true));
            $data = ["Phone" => $request->phone, "Guid" => $request->guid, "OperationNumber" => (int)$request->operation_number, "Invoice" => (int)$request->invoice, "Code" => $code];

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();
            $result = $service->hashDataAndSendMtn($data, 'confirm_invoice', 'pos_web/payment_phone/confirm');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function refundInitiate(Request $request)
    {
        $request->validate([
            'invoice' => 'required|integer',
        ]);
        try {

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();

            $data = ["Invoice" => (int)$request->invoice];

            $result = $service->hashDataAndSendMtn($data, 'refund_inti_invoice', 'pos_web/invoice/refund/initiate');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function refundConfirm(Request $request)
    {
        try {
            $request->validate([
                'base_invoice' => 'required|integer',
                'refund_invoice' => 'required|integer',
            ]);

            $data = ["BaseInvoice" => (int)$request->base_invoice, "RefundInvoice" => (int)$request->refund_invoice];

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();
            $result = $service->hashDataAndSendMtn($data, 'refund_confirm_invoice', 'pos_web/invoice/refund/confirm');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
    public function refundCancel(Request $request)
    {
        $request->validate([
            'invoice_cancel_id' => 'required|integer',
        ]);

        try {

            $service = new MtnApiService(config('app.mtn_terminal_id'));
            $service->set_private_key();

            $data = ["InvoiceCancelID" => (int)$request->invoice];

            $result = $service->hashDataAndSendMtn($data, 'refund_cancel_invoice', 'pos_web/invoice/refund/cancel');

            return response()->json([
                $result['data']
            ]);
        } catch (ValidationException $e) {
            return $this->validationError($e->errors(), 'Validation failed');
        } catch (Exception $e) {
            Log::error('Login error', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return $this->serverError('An error occurred during login. Please try again.');
        }
    }
}
