<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use phpseclib3\Crypt\RSA;

class MtnApiService
{
    protected $baseUrl;
    protected $terminalId;
    protected $privateKey;
    protected $publicKey;
    protected $serialNumber;
    protected $guid;

    public function __construct($terminalId, $serialNumber = null)
    {
        $this->baseUrl = config('app.mtn_base_url');
        $this->terminalId = $terminalId;
        $this->serialNumber = $serialNumber ?? 'TC-' . date('Y');
    }

    public function set_private_key(): void
    {
        $this->privateKey = Storage::get("keys/private_key.pem");
    }

    public function generateKeys($keySize = 1024)
    {
        try {
            $rsa = RSA::createKey($keySize);

            $privateKeyRsax = $rsa->toString('PKCS8');
            $publicKeyRsa = $rsa->getPublicKey()->toString('PKCS8');
            // $this->privateKey = $privateKeyRsa;
            $privateKeystr_replace = str_replace("\r", "", $privateKeyRsax);
            // $this->publicKey = $rsa->getPublicKey()->toString('PKCS8');

            // Storage::put("keys/private_key.pem", $privateKeystr_replace);
            // Storage::put("keys/public_key.pem", $publicKeyRsa);

            $this->privateKey = Storage::get("keys/private_key.pem");
            $this->publicKey = Storage::get("keys/public_key.pem");

            return $this;
        } catch (\Exception $e) {
            throw new \Exception('Failed to generate keys: ' . $e->getMessage());
        }
    }

    protected function formatPublicKey($publicKey)
    {
        return str_replace(
            ["-----BEGIN PUBLIC KEY-----", "-----END PUBLIC KEY-----", "\n", "\r", " "],
            "",
            $publicKey
        );
    }

    protected function sendRequest($endpoint, $data, $requestName, $req)
    {
        $signature = ($data);


        $response = $this->makeHttpRequest($req, $signature, $requestName);

        return [
            'success' => $response['success'],
            'data' => $response['body'],
            'log_id' => $response['success'],
        ];
    }

    public function activateTerminal($activationCode)
    {
        $publicKeyFormatted = $this->formatPublicKey($this->publicKey);

        $data = [
            'Key' => $publicKeyFormatted,
            'Secret' => $activationCode,
            'Serial' => $this->serialNumber
        ];

        return $this->hashDataAndSendMtn($data, 'terminal_activation',  'pos_web/pos/activate');
    }

    public function hashDataAndSendMtn($data, $nameReq, $urlReq)
    {
        $jsonBody = json_encode($data, JSON_UNESCAPED_SLASHES);

        // $private_req = "-----BEGIN PRIVATE KEY-----\nMIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAOu1yOzwJyUhvfOT\nPpWE1XlHd1T3ycEBiVtl7grfEv4mDzRLngfkL2KnvXPtdxrzvKJMPB+CrKUggiRV\n5M4bQ+kG4dAdtUalaAJ/mMj10KqJo0BXyfeaCPFmL9ErWpSD5yfG7/ft68jtLbkM\nXEPCT803YKvxuOwmv4qkWNhdrH6bAgMBAAECgYEA1beHrL264IINlvJlBnDglgxk\neJgLSIK6JjLveM/2ipW7p3EAoqsVA+CPeLNRERD6cjWpNfDn1Pj6xzNSOEakdMFE\ngkggpnUfAcmEseKsVXgLhHCkDamtHNHrscZysSmShlV9SZFZy5VUtEPiyGGfeBv0\nzki8RXkEHf5eLa23vVECQQD/xTuGFHE/WtSbo28EdhYRrOXR2jT4rp9D0OtKNziC\nHnG6XgYVwwjU8I12FVp0zmlj+oZZRG0PF7Fs2nb6Ag81AkEA6+vxcqSWeXmadDyl\nPmIIUEnYBPt6KNDGrtloB1LfHuH7tk4Ruw5R7hHBGFyoZxrvKb1IaFJiK6FN6hM8\nby4AjwJBAPCrwsrzyWx4J78cFob4JuQS8V56GYY3qp3BpOV8+KtXwmQ/vTIVB+Z3\nRGVsymVHJVxyjKDLBmRBQy7Ok03FSlECQA1oyr92ZWsgI0xKCiZikhBmMhYOaSIa\nw8A6cTRBXJc3xwPUqiNUrAl5Dt2gnp3FpSRdMtljHVtXfsM4Nq4nEjcCQDQrVBvK\nHlAZGeze2gJWIo3UgCQ+exqp3u2M4PbUiUtsWRrOZPjpn3M1CN4enhKd2A5Ckigs\nem8EaTgXkjtb61k=\n-----END PRIVATE KEY-----";
        $private_req =  $this->privateKey;
        
        $privateKeyObj = openssl_pkey_get_private($private_req);
        if (!$privateKeyObj) {
            throw new \Exception('Failed to load private key: ' . openssl_error_string());
        }

        $encrypted = '';
        openssl_sign($jsonBody, $encrypted, $privateKeyObj, OPENSSL_ALGO_SHA256);

        $signature = base64_encode($encrypted);
        return $this->sendRequest($nameReq, $signature, $urlReq, $jsonBody);
    }

    protected function makeHttpRequest($data, $signature, $requestName)
    {
        $headers = [
            'Subject: ' . $this->terminalId,
            'Request-Name: ' . $requestName,
            'X-Signature: ' . $signature,
            'Accept-Language: en',
            'Content-Type: application/json',
            'Connection: close'
        ];

        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->baseUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_FOLLOWLOCATION => false,
            CURLOPT_HEADER => false,
            CURLOPT_FAILONERROR => false,
            CURLOPT_ENCODING => '',
            CURLINFO_HEADER_OUT => true,
        ]);

        try {
            $response = curl_exec($ch);

            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            $errorNo = curl_errno($ch);

            if ($errorNo !== 0) {
                throw new \Exception('cURL Error (' . $errorNo . '): ' . $error);
            }

            if ($response === false) {
                throw new \Exception('cURL request failed: ' . $error);
            }

            $responseData = json_decode($response, true) ?: [];

            curl_close($ch);

            return [
                'success' => $httpCode == 200,
                'http_code' => $httpCode,
                'body' => $responseData,
                'error_code' => $responseData['Errno'] ?? null,
                'error_message' => $responseData['Error'] ?? null
            ];
        } catch (\Exception $e) {
            if (isset($ch) && is_resource($ch)) {
                curl_close($ch);
            }

            return [
                'success' => false,
                'http_code' => 0,
                'body' => [],
                'error_code' => 'REQUEST_FAILED',
                'error_message' => $e->getMessage()
            ];
        }
    }
}