<?php
// قم بتشغيل هذا إذا كنت تستخدم composer

use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel\ErrorCorrectionLevelLow;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Logo\Logo;
use Endroid\QrCode\RoundBlockSizeMode\RoundBlockSizeModeMargin;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Writer\ValidationException;

$response = null;
$debug_info = [];

// إعدادات ثابتة
$accessToken = '9|ODmCqOzpVycVJ0aLuJ62ZgYvLTtisXZt5DpjTj6dfcd4f6ce';
$secretKey = 'super-secret-key-123456';
$baseUrl = 'http://192.168.50.8/services/public/api/mtn/v1';

// GUID ثابت للاستخدام
$guid = 'TCS-' . date('YmdHis') . '-' . rand(1000, 9999);

// إعداد مجلد QR Codes
$qrTempDir = 'qrcodes/';
if (!is_dir($qrTempDir)) {
    mkdir($qrTempDir, 0755, true);
}

/**
 * دالة إرسال الطلبات العامة
 */
function sendMTNRequest($endpoint, $payload, $accessToken, $secretKey) {
    global $baseUrl;
    $url = $baseUrl . $endpoint;
    
    // إنشاء التوقيع بالطريقة الصحيحة (JSON)
    $signature = hash_hmac('sha256', json_encode($payload), $secretKey);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . $accessToken,
            'X-SIGNATURE: ' . $signature,
            'Accept: application/json',
            'Content-Type: application/json'
        ],
    ]);
    
    $result = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return [
        'code' => $httpCode,
        'response' => $result,
        'response_json' => json_decode($result, true),
        'signature' => $signature
    ];
}

/**
 * دالة إنشاء صورة QR Code باستخدام Endroid QR Code
 */
function generateQRCodeImage($text) {
    global $qrTempDir;
    
    try {
        // إنشاء اسم فريد للصورة
        $filename = $qrTempDir . 'qr_' . md5($text) . '_' . time() . '.png';
        
        // إنشاء QR Code
        $qrCode = QrCode::create($text)
            ->setEncoding(new Encoding('UTF-8'))
            ->setErrorCorrectionLevel(new ErrorCorrectionLevelLow())
            ->setSize(300)
            ->setMargin(10)
            ->setRoundBlockSizeMode(new RoundBlockSizeModeMargin())
            ->setForegroundColor(new Color(0, 0, 0))
            ->setBackgroundColor(new Color(255, 255, 255));
        
        $writer = new PngWriter();
        $result = $writer->write($qrCode);
        
        // حفظ الصورة
        $result->saveToFile($filename);
        
        return $filename;
        
    } catch (Exception $e) {
        // طريقة بديلة إذا فشلت المكتبة
        return generateSimpleQRCode($text);
    }
}

/**
 * دالة بديلة لإنشاء QR Code باستخدام GD Library
 */
function generateSimpleQRCode($text) {
    global $qrTempDir;
    
    // اسم الملف
    $filename = $qrTempDir . 'qr_simple_' . md5($text) . '_' . time() . '.png';
    
    // طريقة بديلة: إنشاء صورة تحتوي على النص فقط
    $width = 300;
    $height = 300;
    
    $image = imagecreate($width, $height);
    
    // خلفية بيضاء
    $bgColor = imagecolorallocate($image, 255, 255, 255);
    imagefill($image, 0, 0, $bgColor);
    
    // لون النص (أسود)
    $textColor = imagecolorallocate($image, 0, 0, 0);
    
    // إضافة نص "QR Code"
    $font = 5; // حجم الخط الافتراضي في GD
    $textToShow = "QR Code:\n" . substr($text, 0, 50) . "\n" . substr($text, 50, 50);
    
    // كسر النص إلى أسطر
    $lines = explode("\n", wordwrap($textToShow, 40));
    $y = 20;
    
    foreach ($lines as $line) {
        imagestring($image, $font, 10, $y, $line, $textColor);
        $y += 20;
    }
    
    // إضافة إطار
    imagerectangle($image, 0, 0, $width-1, $height-1, $textColor);
    
    // حفظ الصورة
    imagepng($image, $filename);
    imagedestroy($image);
    
    return $filename;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'get_invoice') {
        // ============ Get Invoice ============
        $invoice = (int)($_POST['invoice'] ?? 0);
        
        if ($invoice <= 0) {
            $response = ['error' => 'يرجى إدخال رقم فاتورة صحيح'];
        } else {
            $payload = ['invoice' => $invoice];
            $result = sendMTNRequest('/invoice/get', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/invoice/get',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
                // إنشاء QR Code إذا كان موجوداً
                if (isset($response[0]['Receipt']['Qr']) && !empty($response[0]['Receipt']['Qr'])) {
                    $qrImagePath = generateQRCodeImage($response[0]['Receipt']['Qr']);
                    $response['qr_image_url'] = $qrImagePath;
                }
            } else {
                $response = [
                    'error' => 'فشل الحصول على الفاتورة',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'create_invoice') {
        // ============ Create Invoice ============
        $amount = $_POST['amount'] ?? '';
        $invoice = (int)($_POST['invoice'] ?? 0);
        $ttl = (int)($_POST['ttl'] ?? 30);
        
        // التحقق من البيانات
        $errors = [];
        if (empty($amount) || !is_numeric($amount) || $amount <= 0) {
            $errors[] = 'المبلغ يجب أن يكون رقم موجب';
        }
        if ($invoice <= 0) {
            $errors[] = 'رقم الفاتورة يجب أن يكون رقم موجب';
        }
        if ($ttl <= 0 || $ttl > 1440) {
            $errors[] = 'مدة الصلاحية يجب أن تكون بين 1 و 1440 دقيقة';
        }
        
        if (!empty($errors)) {
            $response = ['error' => implode('، ', $errors)];
        } else {
            $payload = [
                'amount' => (float)$amount,
                'invoice' => $invoice,
                'ttl' => $ttl
            ];
            
            $result = sendMTNRequest('/invoice/create', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/invoice/create',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
                // إنشاء QR Code إذا كان موجوداً
                if (isset($response[0]['Receipt']['Qr']) && !empty($response[0]['Receipt']['Qr'])) {
                    $qrImagePath = generateQRCodeImage($response[0]['Receipt']['Qr']);
                    $response['qr_image_url'] = $qrImagePath;
                }
            } elseif ($result['code'] == 422) {
                $response = [
                    'error' => 'خطأ في التحقق من البيانات',
                    'validation_errors' => $result['response_json']['errors'] ?? [],
                    'response' => $result['response_json']
                ];
            } else {
                $response = [
                    'error' => 'فشل إنشاء الفاتورة',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'payment_initiate') {
        // ============ Payment Initiate ============
        $invoice = (int)($_POST['payment_invoice'] ?? 0);
        $phone_client = $_POST['phone_client'] ?? '';
        $guid = $GLOBALS['guid']; // استخدام GUID الثابت
        
        // التحقق من البيانات
        $errors = [];
        if ($invoice <= 0) {
            $errors[] = 'رقم الفاتورة يجب أن يكون رقم موجب';
        }
        if (empty($phone_client) || !preg_match('/^[0-9]{10,15}$/', $phone_client)) {
            $errors[] = 'رقم الهاتف يجب أن يتكون من 10-15 رقم';
        }
        
        if (!empty($errors)) {
            $response = ['error' => implode('، ', $errors)];
        } else {
            $payload = [
                'invoice' => $invoice,
                'phone_clinte' => $phone_client,
                'guid' => $guid
            ];
            
            $result = sendMTNRequest('/payment/initiate', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/payment/initiate',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
            } elseif ($result['code'] == 422) {
                $response = [
                    'error' => 'خطأ في التحقق من البيانات',
                    'validation_errors' => $result['response_json']['errors'] ?? [],
                    'response' => $result['response_json']
                ];
            } else {
                $response = [
                    'error' => 'فشل بدء عملية الدفع',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'payment_confirm') {
        // ============ Payment Confirm ============
        $phone = $_POST['confirm_phone'] ?? '';
        $guid = $GLOBALS['guid']; // استخدام GUID الثابت
        $operation_number = $_POST['operation_number'] ?? '';
        $invoice = (int)($_POST['confirm_invoice'] ?? 0);
        $code = $_POST['confirm_code'] ?? '';
        
        // التحقق من البيانات
        $errors = [];
        if (empty($phone) || !preg_match('/^[0-9]{10,15}$/', $phone)) {
            $errors[] = 'رقم الهاتف يجب أن يتكون من 10-15 رقم';
        }
        if (empty($operation_number) || !is_numeric($operation_number)) {
            $errors[] = 'رقم العملية مطلوب ويجب أن يكون رقماً';
        }
        if ($invoice <= 0) {
            $errors[] = 'رقم الفاتورة يجب أن يكون رقم موجب';
        }
        if (empty($code) || !preg_match('/^[0-9]{6}$/', $code)) {
            $errors[] = 'كود التأكيد يجب أن يتكون من 6 أرقام';
        }
        
        if (!empty($errors)) {
            $response = ['error' => implode('، ', $errors)];
        } else {
            $payload = [
                'phone' => $phone,
                'guid' => $guid,
                'operation_number' => $operation_number,
                'invoice' => $invoice,
                'code' => $code
            ];
            
            $result = sendMTNRequest('/payment/confirm', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/payment/confirm',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
            } elseif ($result['code'] == 422) {
                $response = [
                    'error' => 'خطأ في التحقق من البيانات',
                    'validation_errors' => $result['response_json']['errors'] ?? [],
                    'response' => $result['response_json']
                ];
            } else {
                $response = [
                    'error' => 'فشل تأكيد عملية الدفع',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'refund_initiate') {
        // ============ Refund Initiate ============
        $invoice = (int)($_POST['refund_invoice'] ?? 0);
        
        if ($invoice <= 0) {
            $response = ['error' => 'يرجى إدخال رقم فاتورة صحيح'];
        } else {
            $payload = ['invoice' => $invoice];
            $result = sendMTNRequest('/refund/initiate', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/refund/initiate',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
            } else {
                $response = [
                    'error' => 'فشل بدء عملية الاسترداد',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'refund_confirm') {
        // ============ Refund Confirm ============
        $base_invoice = (int)($_POST['base_invoice'] ?? 0);
        $refund_invoice = (int)($_POST['refund_invoice_confirm'] ?? 0);
        
        // التحقق من البيانات
        $errors = [];
        if ($base_invoice <= 0) {
            $errors[] = 'رقم الفاتورة الأساسية مطلوب';
        }
        if ($refund_invoice <= 0) {
            $errors[] = 'رقم فاتورة الاسترداد مطلوب';
        }
        
        if (!empty($errors)) {
            $response = ['error' => implode('، ', $errors)];
        } else {
            $payload = [
                'base_invoice' => $base_invoice,
                'refund_invoice' => $refund_invoice
            ];
            
            $result = sendMTNRequest('/refund/confirm', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/refund/confirm',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
            } elseif ($result['code'] == 422) {
                $response = [
                    'error' => 'خطأ في التحقق من البيانات',
                    'validation_errors' => $result['response_json']['errors'] ?? [],
                    'response' => $result['response_json']
                ];
            } else {
                $response = [
                    'error' => 'فشل تأكيد عملية الاسترداد',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
        
    } elseif ($action === 'refund_cancel') {
        // ============ Refund Cancel ============
        $invoice_cancel_id = (int)($_POST['invoice_cancel_id'] ?? 0);
        
        if ($invoice_cancel_id <= 0) {
            $response = ['error' => 'يرجى إدخال رقم فاتورة إلغاء صحيح'];
        } else {
            $payload = ['invoice_cancel_id' => $invoice_cancel_id];
            $result = sendMTNRequest('/refund/cancel', $payload, $accessToken, $secretKey);
            
            $debug_info = [
                'endpoint' => '/refund/cancel',
                'payload' => $payload,
                'signature' => $result['signature'],
                'http_code' => $result['code']
            ];
            
            if ($result['code'] == 200) {
                $response = $result['response_json'];
            } else {
                $response = [
                    'error' => 'فشل إلغاء عملية الاسترداد',
                    'http_code' => $result['code'],
                    'response' => $result['response_json']
                ];
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MTN Payment - نظام الفواتير والمدفوعات المتكامل</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        
        header {
            background: linear-gradient(to right, #4f46e5, #7c3aed);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }
        
        header p {
            opacity: 0.9;
            font-size: 16px;
        }
        
        .tabs {
            display: flex;
            background: #f8fafc;
            border-bottom: 2px solid #e2e8f0;
            flex-wrap: wrap;
        }
        
        .tab {
            flex: 1;
            padding: 20px;
            text-align: center;
            background: none;
            border: none;
            font-size: 18px;
            font-weight: 600;
            color: #64748b;
            cursor: pointer;
            transition: all 0.3s;
            border-bottom: 3px solid transparent;
            min-width: 200px;
        }
        
        .tab:hover {
            background: #f1f5f9;
            color: #475569;
        }
        
        .tab.active {
            color: #4f46e5;
            border-bottom-color: #4f46e5;
            background: white;
        }
        
        .tab-content {
            display: none;
            padding: 40px;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .form-container {
            max-width: 800px;
            margin: 0 auto;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 10px;
            color: #334155;
            font-weight: 600;
            font-size: 16px;
        }
        
        .form-group input {
            width: 100%;
            padding: 15px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            font-size: 16px;
            transition: all 0.3s;
            background: #f8fafc;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #4f46e5;
            background: white;
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }
        
        .form-group .hint {
            margin-top: 8px;
            color: #64748b;
            font-size: 14px;
        }
        
        .btn {
            display: block;
            width: 100%;
            padding: 18px;
            background: linear-gradient(to right, #4f46e5, #7c3aed);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 18px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            margin-top: 10px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(79, 70, 229, 0.3);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .btn-secondary {
            background: linear-gradient(to right, #6b7280, #4b5563);
        }
        
        .btn-secondary:hover {
            box-shadow: 0 10px 25px rgba(107, 114, 128, 0.3);
        }
        
        .response-container {
            margin-top: 40px;
            background: #f8fafc;
            border-radius: 15px;
            padding: 25px;
            border: 2px solid #e2e8f0;
        }
        
        .response-title {
            color: #334155;
            margin-bottom: 20px;
            font-size: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .response-title.success {
            color: #059669;
        }
        
        .response-title.error {
            color: #dc2626;
        }
        
        pre {
            background: #1e293b;
            color: #e2e8f0;
            padding: 20px;
            border-radius: 10px;
            overflow-x: auto;
            font-size: 14px;
            line-height: 1.5;
            max-height: 500px;
            overflow-y: auto;
        }
        
        .qr-container {
            text-align: center;
            margin: 20px 0;
            padding: 25px;
            background: white;
            border-radius: 10px;
            border: 2px dashed #cbd5e1;
        }
        
        .qr-code {
            font-family: monospace;
            background: #f1f5f9;
            padding: 15px;
            border-radius: 8px;
            margin: 15px 0;
            word-break: break-all;
            font-size: 12px;
            line-height: 1.4;
            cursor: pointer;
            transition: background 0.3s;
        }
        
        .qr-code:hover {
            background: #e2e8f0;
        }
        
        .qr-image {
            max-width: 300px;
            margin: 20px auto;
            padding: 15px;
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            border: 1px solid #e2e8f0;
        }
        
        .qr-image img {
            width: 100%;
            height: auto;
            display: block;
            border-radius: 5px;
        }
        
        .invoice-details {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
            border: 1px solid #e2e8f0;
        }
        
        .detail-row {
            display: flex;
            margin-bottom: 12px;
            padding-bottom: 12px;
            border-bottom: 1px solid #f1f5f9;
        }
        
        .detail-label {
            width: 200px;
            font-weight: 600;
            color: #475569;
        }
        
        .detail-value {
            flex: 1;
            color: #1e293b;
        }
        
        .status-badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        .status-pending {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-paid {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-expired {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .debug-info {
            margin-top: 30px;
            background: #fffbeb;
            border: 1px solid #fde68a;
            border-radius: 10px;
            padding: 20px;
        }
        
        .debug-toggle {
            background: #f59e0b;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            margin-bottom: 15px;
        }
        
        .debug-content {
            display: none;
        }
        
        .debug-content.show {
            display: block;
        }
        
        .two-column {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        .guid-info {
            background: #f0f9ff;
            border: 2px solid #bae6fd;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .guid-info span {
            font-weight: 600;
            color: #0369a1;
        }
        
        .action-buttons {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .action-buttons .btn {
            flex: 1;
        }
        
        .info-box {
            background: #f8fafc;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .info-box h4 {
            color: #4f46e5;
            margin-bottom: 10px;
            font-size: 16px;
        }
        
        .info-box p {
            color: #64748b;
            font-size: 14px;
            line-height: 1.5;
        }
        
        @media (max-width: 768px) {
            .container {
                border-radius: 10px;
            }
            
            header {
                padding: 20px;
            }
            
            header h1 {
                font-size: 24px;
            }
            
            .tab-content {
                padding: 20px;
            }
            
            .detail-row {
                flex-direction: column;
            }
            
            .detail-label {
                width: 100%;
                margin-bottom: 5px;
            }
            
            .two-column {
                grid-template-columns: 1fr;
            }
            
            .tab {
                min-width: 150px;
                padding: 15px;
                font-size: 16px;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .qr-image {
                max-width: 250px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1>🌐 نظام فواتير MTN المتكامل</h1>
            <p>إنشاء فواتير، مدفوعات، استرداد، وإدارة كاملة</p>
            <div class="guid-info">
                🔑 <strong>معرف الجلسة:</strong> <span><?php echo $guid; ?></span>
            </div>
        </header>
        
        <div class="tabs">
            <button class="tab active" onclick="switchTab('get')">🔍 استعلام فاتورة</button>
            <button class="tab" onclick="switchTab('create')">➕ إنشاء فاتورة</button>
            <button class="tab" onclick="switchTab('payment_init')">💳 بدء الدفع</button>
            <button class="tab" onclick="switchTab('payment_confirm')">✅ تأكيد الدفع</button>
            <button class="tab" onclick="switchTab('refund_init')">🔄 بدء استرداد</button>
            <button class="tab" onclick="switchTab('refund_confirm')">✓ تأكيد استرداد</button>
            <button class="tab" onclick="switchTab('refund_cancel')">❌ إلغاء استرداد</button>
        </div>
        
        <!-- تبويب استعلام الفاتورة -->
        <div id="get" class="tab-content active">
            <div class="form-container">
                <h2 style="color: #4f46e5; margin-bottom: 30px; text-align: center;">استعلام عن فاتورة</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="get_invoice">
                    
                    <div class="form-group">
                        <label for="get_invoice">رقم الفاتورة:</label>
                        <input type="number" 
                               id="get_invoice" 
                               name="invoice" 
                               placeholder="أدخل رقم الفاتورة (مثال: 200)" 
                               required
                               min="1">
                        <div class="hint">أدخل رقم الفاتورة الذي تم إنشاؤه مسبقاً</div>
                    </div>
                    
                    <button type="submit" class="btn">🔍 استعلام عن الفاتورة</button>
                </form>
            </div>
        </div>
        
        <!-- تبويب إنشاء الفاتورة -->
        <div id="create" class="tab-content">
            <div class="form-container">
                <h2 style="color: #7c3aed; margin-bottom: 30px; text-align: center;">إنشاء فاتورة جديدة</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="create_invoice">
                    
                    <div class="info-box">
                        <h4>💡 ملاحظة هامة:</h4>
                        <p>المبلغ يُدخل بالليرة السورية (مثال: 0.2 = 20 ليرة سورية). يتم إنشاء QR Code تلقائياً عند نجاح العملية.</p>
                    </div>
                    
                    <div class="form-group">
                        <label for="amount">المبلغ (ليرة سورية):</label>
                        <input type="number" 
                               step="0.01" 
                               id="amount" 
                               name="amount" 
                               placeholder="مثال: 0.2" 
                               required
                               min="0.01">
                        <div class="hint">المبلغ بالليرة السورية (مثال: 0.2 = 20 ليرة)</div>
                    </div>
                    
                    <div class="form-group">
                        <label for="create_invoice">رقم الفاتورة:</label>
                        <input type="number" 
                               id="create_invoice" 
                               name="invoice" 
                               placeholder="أدخل رقم فريد للفاتورة" 
                               required
                               min="1">
                        <div class="hint">يجب أن يكون رقم فريد غير مكرر (مثال: 202)</div>
                    </div>
                    
                    <div class="form-group">
                        <label for="ttl">مدة الصلاحية (دقائق):</label>
                        <input type="number" 
                               id="ttl" 
                               name="ttl" 
                               placeholder="مثال: 50" 
                               value="30"
                               min="1"
                               max="1440">
                        <div class="hint">الافتراضي 30 دقيقة، الحد الأقصى 1440 دقيقة (24 ساعة)</div>
                    </div>
                    
                    <button type="submit" class="btn">➕ إنشاء الفاتورة</button>
                </form>
            </div>
        </div>
        
        <!-- تبويب بدء الدفع -->
        <div id="payment_init" class="tab-content">
            <div class="form-container">
                <h2 style="color: #10b981; margin-bottom: 30px; text-align: center;">بدء عملية الدفع</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="payment_initiate">
                    
                    <div class="info-box">
                        <h4>📝 معلومات مهمة:</h4>
                        <p>يتم استخدام نفس معرف الجلسة لجميع العمليات لتسهيل التتبع. رقم الهاتف يجب أن يكون بتنسيق سوري (مثال: 963950514975).</p>
                    </div>
                    
                    <div class="form-group">
                        <label for="payment_invoice">رقم الفاتورة:</label>
                        <input type="number" 
                               id="payment_invoice" 
                               name="payment_invoice" 
                               placeholder="أدخل رقم الفاتورة" 
                               required
                               min="1">
                        <div class="hint">رقم الفاتورة المراد الدفع لها</div>
                    </div>
                    
                    <div class="form-group">
                        <label for="phone_client">رقم هاتف العميل:</label>
                        <input type="text" 
                               id="phone_client" 
                               name="phone_client" 
                               placeholder="مثال: 963950514975" 
                               required
                               pattern="[0-9]{10,15}">
                        <div class="hint">رقم الهاتف يجب أن يتكون من 10-15 رقم</div>
                    </div>
                    
                    <div class="guid-info">
                        🔑 <strong>معرف GUID المستخدم:</strong> <span><?php echo $guid; ?></span>
                    </div>
                    
                    <div class="action-buttons">
                        <button type="submit" class="btn">💳 بدء عملية الدفع</button>
                        <button type="button" class="btn btn-secondary" onclick="copyGuid()">📋 نسخ GUID</button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- تبويب تأكيد الدفع -->
        <div id="payment_confirm" class="tab-content">
            <div class="form-container">
                <h2 style="color: #059669; margin-bottom: 30px; text-align: center;">تأكيد عملية الدفع</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="payment_confirm">
                    
                    <div class="guid-info">
                        🔑 <strong>معرف GUID المستخدم:</strong> <span><?php echo $guid; ?></span>
                    </div>
                    
                    <div class="two-column">
                        <div class="form-group">
                            <label for="confirm_phone">رقم الهاتف:</label>
                            <input type="text" 
                                   id="confirm_phone" 
                                   name="confirm_phone" 
                                   placeholder="مثال: 963950514975" 
                                   required
                                   pattern="[0-9]{10,15}">
                        </div>
                        
                        <div class="form-group">
                            <label for="confirm_invoice">رقم الفاتورة:</label>
                            <input type="number" 
                                   id="confirm_invoice" 
                                   name="confirm_invoice" 
                                   placeholder="مثال: 200" 
                                   required
                                   min="1">
                        </div>
                    </div>
                    
                    <div class="two-column">
                        <div class="form-group">
                            <label for="operation_number">رقم العملية:</label>
                            <input type="text" 
                                   id="operation_number" 
                                   name="operation_number" 
                                   placeholder="مثال: 7007531401826869" 
                                   required
                                   pattern="[0-9]+">
                        </div>
                        
                        <div class="form-group">
                            <label for="confirm_code">كود التأكيد:</label>
                            <input type="text" 
                                   id="confirm_code" 
                                   name="confirm_code" 
                                   placeholder="أدخل الكود المكون من 6 أرقام" 
                                   required
                                   pattern="[0-9]{6}"
                                   maxlength="6">
                        </div>
                    </div>
                    
                    <div class="action-buttons">
                        <button type="submit" class="btn">✅ تأكيد عملية الدفع</button>
                        <button type="button" class="btn btn-secondary" onclick="generateRandomCode()">🎲 توليد كود عشوائي</button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- تبويب بدء الاسترداد -->
        <div id="refund_init" class="tab-content">
            <div class="form-container">
                <h2 style="color: #f59e0b; margin-bottom: 30px; text-align: center;">بدء عملية الاسترداد</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="refund_initiate">
                    
                    <div class="info-box">
                        <h4>⚠️ تنبيه:</h4>
                        <p>يجب أن تكون الفاتورة مدفوعة ومازالت صالحة للاسترداد. بعض العمليات قد تتضمن رسوم استرداد.</p>
                    </div>
                    
                    <div class="form-group">
                        <label for="refund_invoice">رقم الفاتورة:</label>
                        <input type="number" 
                               id="refund_invoice" 
                               name="refund_invoice" 
                               placeholder="أدخل رقم الفاتورة المراد استردادها" 
                               required
                               min="1">
                        <div class="hint">رقم الفاتورة التي تم دفعها مسبقاً</div>
                    </div>
                    
                    <button type="submit" class="btn">🔄 بدء عملية الاسترداد</button>
                </form>
            </div>
        </div>
        
        <!-- تبويب تأكيد الاسترداد -->
        <div id="refund_confirm" class="tab-content">
            <div class="form-container">
                <h2 style="color: #3b82f6; margin-bottom: 30px; text-align: center;">تأكيد عملية الاسترداد</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="refund_confirm">
                    
                    <div class="two-column">
                        <div class="form-group">
                            <label for="base_invoice">رقم الفاتورة الأساسية:</label>
                            <input type="number" 
                                   id="base_invoice" 
                                   name="base_invoice" 
                                   placeholder="مثال: 200" 
                                   required
                                   min="1">
                            <div class="hint">رقم الفاتورة الأصلية</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="refund_invoice_confirm">رقم فاتورة الاسترداد:</label>
                            <input type="number" 
                                   id="refund_invoice_confirm" 
                                   name="refund_invoice_confirm" 
                                   placeholder="مثال: 201" 
                                   required
                                   min="1">
                            <div class="hint">رقم فاتورة الاسترداد الذي تم إنشاؤه</div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn">✓ تأكيد عملية الاسترداد</button>
                </form>
            </div>
        </div>
        
        <!-- تبويب إلغاء الاسترداد -->
        <div id="refund_cancel" class="tab-content">
            <div class="form-container">
                <h2 style="color: #dc2626; margin-bottom: 30px; text-align: center;">إلغاء عملية الاسترداد</h2>
                
                <form method="post">
                    <input type="hidden" name="action" value="refund_cancel">
                    
                    <div class="info-box">
                        <h4>❌ تحذير:</h4>
                        <p>عملية الإلغاء لا يمكن التراجع عنها. تأكد من رقم فاتورة الاسترداد قبل الإلغاء.</p>
                    </div>
                    
                    <div class="form-group">
                        <label for="invoice_cancel_id">رقم فاتورة الإلغاء:</label>
                        <input type="number" 
                               id="invoice_cancel_id" 
                               name="invoice_cancel_id" 
                               placeholder="أدخل رقم فاتورة الإلغاء" 
                               required
                               min="1">
                        <div class="hint">رقم فاتورة الاسترداد المراد إلغاؤها</div>
                    </div>
                    
                    <button type="submit" class="btn">❌ إلغاء عملية الاسترداد</button>
                </form>
            </div>
        </div>
        
        <!-- عرض النتائج -->
        <?php if ($response): ?>
            <div class="response-container">
                <?php if (isset($response['error'])): ?>
                    <div class="response-title error">
                        ❌ <?php echo $response['error']; ?>
                    </div>
                    
                    <?php if (isset($response['http_code'])): ?>
                        <div style="background: #fee2e2; padding: 15px; border-radius: 8px; margin: 15px 0;">
                            <strong>رمز HTTP:</strong> <?php echo $response['http_code']; ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (isset($response['validation_errors'])): ?>
                        <div style="background: #fef3c7; padding: 15px; border-radius: 8px; margin: 15px 0;">
                            <strong>أخطاء التحقق:</strong>
                            <pre style="background: transparent; color: #92400e;"><?php 
                                print_r($response['validation_errors']); 
                            ?></pre>
                        </div>
                    <?php endif; ?>
                    
                <?php else: ?>
                    <div class="response-title success">
                        ✅ نجاح العملية
                    </div>
                    
                    <?php if (isset($response[0]['Receipt'])): ?>
                        <?php $receipt = $response[0]['Receipt']; ?>
                        
                        <div class="invoice-details">
                            <div class="detail-row">
                                <div class="detail-label">رقم الفاتورة:</div>
                                <div class="detail-value"><?php echo $receipt['Invoice'] ?? 'غير معروف'; ?></div>
                            </div>
                            
                            <?php if (isset($receipt['Amount'])): ?>
                                <div class="detail-row">
                                    <div class="detail-label">المبلغ:</div>
                                    <div class="detail-value"><?php echo ($receipt['Amount'] / 100) . ' ليرة'; ?></div>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($receipt['Status'])): ?>
                                <div class="detail-row">
                                    <div class="detail-label">الحالة:</div>
                                    <div class="detail-value">
                                        <?php 
                                        $status = $receipt['Status'] ?? 0;
                                        $statusText = match($status) {
                                            1 => '<span class="status-badge status-pending">معلقة</span>',
                                            2 => '<span class="status-badge status-paid">مدفوعة</span>',
                                            3 => '<span class="status-badge status-expired">منتهية</span>',
                                            4 => '<span class="status-badge">ملغاة</span>',
                                            default => '<span class="status-badge">غير معروف</span>'
                                        };
                                        echo $statusText . ' (كود: ' . $status . ')';
                                        ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($receipt['Transaction'])): ?>
                                <div class="detail-row">
                                    <div class="detail-label">رقم المعاملة:</div>
                                    <div class="detail-value"><?php echo $receipt['Transaction']; ?></div>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($receipt['Paid'])): ?>
                                <div class="detail-row">
                                    <div class="detail-label">مدفوعة:</div>
                                    <div class="detail-value"><?php echo $receipt['Paid'] ? '✅ نعم' : '❌ لا'; ?></div>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($receipt['Qr'])): ?>
                                <div class="qr-container">
                                    <h3 style="color: #4f46e5; margin-bottom: 15px;">📱 كود QR للدفع</h3>
                                    
                                    <?php if (isset($response['qr_image_url']) && file_exists($response['qr_image_url'])): ?>
                                        <div class="qr-image">
                                            <img src="<?php echo $response['qr_image_url']; ?>" alt="QR Code" onerror="this.onerror=null; this.style.display='none';">
                                        </div>
                                        <p style="color: #64748b; font-size: 14px; margin-top: 10px;">
                                            يمكن مسح هذا الكود من تطبيق MTN للدفع
                                        </p>
                                    <?php else: ?>
                                        <div style="background: #fef3c7; padding: 15px; border-radius: 8px; margin: 15px 0;">
                                            <p style="color: #92400e;">⚠️ لم يتم إنشاء صورة QR Code. يمكن نسخ النص أدناه.</p>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="qr-code" onclick="copyText(this)">
                                        <?php echo htmlspecialchars($receipt['Qr']); ?>
                                    </div>
                                    <p style="color: #64748b; font-size: 12px;">
                                        اضغط على النص أعلاه لنسخ كود QR
                                    </p>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                    <?php elseif (isset($response[0]['Errno'])): ?>
                        <div style="background: #d1fae5; padding: 15px; border-radius: 8px; margin: 15px 0;">
                            <strong>رقم الخطأ:</strong> <?php echo $response[0]['Errno']; ?>
                        </div>
                        
                        <?php if (isset($response[0]['OperationNumber'])): ?>
                            <div style="background: #dbeafe; padding: 15px; border-radius: 8px; margin: 15px 0;">
                                <strong>رقم العملية:</strong> <?php echo $response[0]['OperationNumber']; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (isset($response[0]['Transaction'])): ?>
                            <div style="background: #d1fae5; padding: 15px; border-radius: 8px; margin: 15px 0;">
                                <strong>رقم المعاملة:</strong> <?php echo $response[0]['Transaction']; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (isset($response[0]['Parameters'])): ?>
                            <div class="invoice-details">
                                <h3 style="color: #475569; margin-bottom: 15px;">📋 معلومات المستفيد</h3>
                                <?php foreach ($response[0]['Parameters'] as $param): ?>
                                    <div class="detail-row">
                                        <div class="detail-label"><?php echo $param['Name'] ?? $param['ParameterName']; ?>:</div>
                                        <div class="detail-value"><?php echo $param['Data']; ?></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (isset($response[0]['RefundInvoice'])): ?>
                            <div style="background: #f0f9ff; padding: 15px; border-radius: 8px; margin: 15px 0;">
                                <strong>فاتورة الاسترداد:</strong> <?php echo $response[0]['RefundInvoice']; ?>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                    
                    <div class="action-buttons" style="margin-top: 20px;">
                        <button type="button" class="btn" onclick="toggleResponse()">📄 عرض/إخفاء الاستجابة الكاملة</button>
                        <button type="button" class="btn btn-secondary" onclick="downloadResponse()">💾 تحميل الاستجابة</button>
                    </div>
                    
                    <div id="fullResponse" style="display: none;">
                        <h3 style="color: #475569; margin: 25px 0 15px;">📄 الاستجابة الكاملة:</h3>
                        <pre><?php print_r($response); ?></pre>
                    </div>
                <?php endif; ?>
                
                <!-- معلومات التصحيح -->
                <?php if (!empty($debug_info)): ?>
                    <div class="debug-info">
                        <button class="debug-toggle" onclick="toggleDebug()">🐛 عرض معلومات التصحيح</button>
                        <div class="debug-content" id="debugContent">
                            <pre><?php print_r($debug_info); ?></pre>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <!-- معلومات الاتصال -->
        <div style="padding: 20px; background: #f8fafc; border-top: 2px solid #e2e8f0; text-align: center; color: #64748b; font-size: 14px;">
            <p><strong>📡 معلومات الاتصال:</strong> 
                API: <?php echo $baseUrl; ?> | 
                Token: <?php echo substr($accessToken, 0, 10) . '...'; ?>
            </p>
            <p>🕐 وقت النظام: <?php echo date('Y-m-d H:i:s'); ?> | 🔑 الجلسة: <?php echo $guid; ?></p>
        </div>
    </div>
    
    <script>
        // تبديل التبويبات
        function switchTab(tabName) {
            // إخفاء جميع المحتويات
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // إزالة النشط من جميع الأزرار
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // إظهار المحتوى المطلوب
            document.getElementById(tabName).classList.add('active');
            
            // جعل الزر نشط
            event.target.classList.add('active');
            
            // توليد بيانات عشوائية للاختبار
            populateTestData(tabName);
        }
        
        // تبديل عرض معلومات التصحيح
        function toggleDebug() {
            const debugContent = document.getElementById('debugContent');
            debugContent.classList.toggle('show');
        }
        
        // تبديل عرض الاستجابة الكاملة
        function toggleResponse() {
            const fullResponse = document.getElementById('fullResponse');
            fullResponse.style.display = fullResponse.style.display === 'none' ? 'block' : 'none';
        }
        
        // تعبئة تلقائية للحقول
        document.addEventListener('DOMContentLoaded', function() {
            // تعبئة حقول المثال
            const urlParams = new URLSearchParams(window.location.search);
            const invoice = urlParams.get('invoice');
            
            if (invoice) {
                document.getElementById('get_invoice').value = invoice;
            }
            
            // توليد بيانات عشوائية للاختبار
            populateTestData('create');
        });
        
        // التحقق من صحة البيانات قبل الإرسال
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                const action = this.querySelector('input[name="action"]').value;
                let isValid = true;
                
                if (action === 'create_invoice') {
                    const amount = parseFloat(this.querySelector('input[name="amount"]').value);
                    const invoice = parseInt(this.querySelector('input[name="invoice"]').value);
                    const ttl = parseInt(this.querySelector('input[name="ttl"]').value);
                    
                    if (amount <= 0) {
                        alert('يرجى إدخال مبلغ صحيح أكبر من صفر');
                        isValid = false;
                    }
                    
                    if (invoice <= 0) {
                        alert('يرجى إدخال رقم فاتورة صحيح');
                        isValid = false;
                    }
                    
                    if (ttl <= 0 || ttl > 1440) {
                        alert('مدة الصلاحية يجب أن تكون بين 1 و 1440 دقيقة');
                        isValid = false;
                    }
                    
                } else if (action === 'payment_initiate') {
                    const phone = this.querySelector('input[name="phone_client"]').value;
                    if (!/^[0-9]{10,15}$/.test(phone)) {
                        alert('رقم الهاتف يجب أن يتكون من 10-15 رقم');
                        isValid = false;
                    }
                    
                } else if (action === 'payment_confirm') {
                    const phone = this.querySelector('input[name="confirm_phone"]').value;
                    const code = this.querySelector('input[name="confirm_code"]').value;
                    
                    if (!/^[0-9]{10,15}$/.test(phone)) {
                        alert('رقم الهاتف يجب أن يتكون من 10-15 رقم');
                        isValid = false;
                    }
                    
                    if (!/^[0-9]{6}$/.test(code)) {
                        alert('كود التأكيد يجب أن يتكون من 6 أرقام');
                        isValid = false;
                    }
                }
                
                if (!isValid) {
                    e.preventDefault();
                }
            });
        });
        
        // وظيفة لملء بيانات الاختبار تلقائياً
        function populateTestData(tabName) {
            const now = new Date();
            
            switch(tabName) {
                case 'create':
                    const randomInvoice = Math.floor(Math.random() * 9000) + 1000;
                    document.getElementById('create_invoice').value = randomInvoice;
                    break;
                    
                case 'payment_init':
                    document.getElementById('payment_invoice').value = Math.floor(Math.random() * 9000) + 1000;
                    break;
                    
                case 'payment_confirm':
                    document.getElementById('operation_number').value = '70075314' + 
                        String(Math.floor(Math.random() * 10000000)).padStart(7, '0');
                    document.getElementById('confirm_code').value = String(Math.floor(Math.random() * 900000) + 100000);
                    break;
                    
                case 'refund_confirm':
                    document.getElementById('base_invoice').value = Math.floor(Math.random() * 9000) + 1000;
                    document.getElementById('refund_invoice_confirm').value = Math.floor(Math.random() * 9000) + 1001;
                    break;
            }
        }
        
        // نسخ النص
        function copyText(element) {
            const text = element.textContent;
            navigator.clipboard.writeText(text).then(() => {
                alert('تم نسخ النص إلى الحافظة');
            });
        }
        
        // نسخ GUID
        function copyGuid() {
            const guidElements = document.querySelectorAll('.guid-info span');
            if (guidElements.length > 0) {
                const guid = guidElements[0].textContent;
                navigator.clipboard.writeText(guid).then(() => {
                    alert('تم نسخ GUID إلى الحافظة');
                });
            }
        }
        
        // توليد كود عشوائي
        function generateRandomCode() {
            const code = String(Math.floor(Math.random() * 900000) + 100000);
            document.getElementById('confirm_code').value = code;
        }
        
        // تحميل الاستجابة كملف
        function downloadResponse() {
            const response = <?php echo json_encode($response ?? []); ?>;
            const dataStr = JSON.stringify(response, null, 2);
            const dataUri = 'data:application/json;charset=utf-8,'+ encodeURIComponent(dataStr);
            
            const exportFileDefaultName = 'mtn_response_' + new Date().toISOString().slice(0,10) + '.json';
            
            const linkElement = document.createElement('a');
            linkElement.setAttribute('href', dataUri);
            linkElement.setAttribute('download', exportFileDefaultName);
            linkElement.click();
        }
        
        // عرض رسالة تأكيد قبل إرسال النماذج الحساسة
        document.querySelectorAll('form').forEach(form => {
            const action = form.querySelector('input[name="action"]').value;
            if (action === 'refund_cancel') {
                form.addEventListener('submit', function(e) {
                    if (!confirm('⚠️ هل أنت متأكد من إلغاء عملية الاسترداد؟ لا يمكن التراجع عن هذه العملية.')) {
                        e.preventDefault();
                    }
                });
            }
        });
        
        // إعادة تحميل الصورة إذا فشلت
        document.querySelectorAll('img[alt="QR Code"]').forEach(img => {
            img.addEventListener('error', function() {
                this.style.display = 'none';
                const container = this.closest('.qr-container');
                if (container) {
                    const warning = document.createElement('div');
                    warning.style.background = '#fef3c7';
                    warning.style.padding = '15px';
                    warning.style.borderRadius = '8px';
                    warning.style.margin = '15px 0';
                    warning.innerHTML = '<p style="color: #92400e;">⚠️ لم يتم تحميل صورة QR Code. يمكن نسخ النص أدناه.</p>';
                    container.insertBefore(warning, this.nextSibling);
                }
            });
        });
    </script>
</body>
</html>